namespace Tests\Feature;

use App\Models\User;
use App\Models\Product;
use App\Models\Category;
use App\Models\Brand;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class SellerProductManagementTest extends TestCase
{
    use RefreshDatabase;

    public function test_seller_can_view_their_products()
    {
        $seller = User::factory()->create(['role' => 'seller']);
        $this->actingAs($seller);

        $product = Product::factory()->create(['seller_id' => $seller->id]);

        $response = $this->get(route('seller.products.index'));

        $response->assertStatus(200);
        $response->assertSee($product->name);
    }

    public function test_seller_can_create_a_product()
    {
        $seller = User::factory()->create(['role' => 'seller']);
        $this->actingAs($seller);

        $category = Category::factory()->create();
        $brand = Brand::factory()->create();

        $response = $this->post(route('seller.products.store'), [
            'name' => 'Test Product',
            'category_id' => $category->id,
            'brand_id' => $brand->id,
            'description' => 'Test Description',
        ]);

        $response->assertRedirect(route('seller.products.index'));
        $this->assertDatabaseHas('products', [
            'name' => 'Test Product',
            'seller_id' => $seller->id,
        ]);
    }

    public function test_seller_can_edit_their_product()
    {
        $seller = User::factory()->create(['role' => 'seller']);
        $this->actingAs($seller);

        $product = Product::factory()->create(['seller_id' => $seller->id]);

        $response = $this->patch(route('seller.products.update', $product), [
            'name' => 'Updated Product Name',
        ]);

        $response->assertRedirect(route('seller.products.index'));
        $this->assertDatabaseHas('products', [
            'id' => $product->id,
            'name' => 'Updated Product Name',
        ]);
    }

    public function test_seller_can_delete_their_product()
    {
        $seller = User::factory()->create(['role' => 'seller']);
        $this->actingAs($seller);

        $product = Product::factory()->create(['seller_id' => $seller->id]);

        $response = $this->delete(route('seller.products.destroy', $product));

        $response->assertRedirect(route('seller.products.index'));
        $this->assertDatabaseMissing('products', [
            'id' => $product->id,
        ]);
    }

    public function test_seller_cannot_manage_other_sellers_products()
    {
        $seller = User::factory()->create(['role' => 'seller']);
        $this->actingAs($seller);

        $otherSeller = User::factory()->create(['role' => 'seller']);
        $product = Product::factory()->create(['seller_id' => $otherSeller->id]);

        $response = $this->patch(route('seller.products.update', $product), [
            'name' => 'Unauthorized Update',
        ]);

        $response->assertStatus(403);
    }
}